format short e

skip = 0;         % Help variable which is set to 1 if user selects
                  % QUIT from main menu. If skip = 1, the last part of
                  % ACLIN, where the actual linearization takes place,
                  % is skipped.

% Welcome...
% ----------
clc
disp('FDC toolbox - ACLIN');
disp('===================');
disp('This program will linearize the nonlinear aircraft model in SIMULINK.');
disp(' ');

% Enter name of the aircraft model.
% ---------------------------------
disp('Enter name of the aircraft model in Simulink (default: BEAVER)');
sysname = input('> ','s');
if isempty(sysname)
   sysname = 'beaver';
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Define or load operating point.
% ===============================

ok = 0;
while ok ~= 1  % While-loop for correctly entering or loading operating
               % point. Normally, loop will be quitted directly, but if
               % definitions are not OK, operating point definition will
               % start all over again. If ok == 1, while-loop will be
               % left, and ACLIN will proceed with model parameter defi-
               % nition and linearization itself.
               % -------------------------------------------------------
   clc
   opt = txtmenu('Choose one of the following options',...
                 'Load operating point from file',...
                 'Manually define operating point',...
                 'Use operating point defined in workspace',...
                 'Run ACTRIM to determine trimmed flight condition',...
                 'Quit');

   if opt == 1                                       % LOAD OPERATING POINT
                                                     % --------------------
      triload;
      ok = 1;
      
   elseif opt == 2                        % MANUALLY DEFINE OPERATING POINT
                                          % -------------------------------
      clc
      disp('Manually define operating point.');
      disp('--------------------------------');

      % While-loop for correctly entering the state vector.
      % ---------------------------------------------------
      ok1 = 0;        % If ok1 == 1, the while-loop for entering xinco
                      % will be quitted; proceed with definition of uaero0.

      proceed = 1;    % If proceed == 0, the while-loop for entering xinco
                      % will be quitted, and manual definition will stop.

      while ok1 ~= 1 & proceed == 1
         disp(' ');
         disp('State vector: [V alpha beta p q r psi theta phi xe ye H]''');
         disp(' ');

         % While-loop to make sure that V has a value unequal to zero. User
         % must select a reasonable V himself, but at least this check makes
         % sure that no 'Division by zero errors' will occur in the Simulink
         % system of the nonlinear aircraft model.
         % -----------------------------------------------------------------
         ok2 = 0;
         while ok2 ~= 1
            V = input('airspeed [m/s]          : ');
            if isempty(V) | V == 0
               disp(' ');
               disp('V must have a value > 0!');
            else
               ok2 = 1;
            end
         end

         alpha = input('angle of attack [rad], default = 0: '); if isempty(alpha), alpha = 0; end;
         beta  = input('sideslip angle [rad], default = 0: ');  if isempty(beta) , beta  = 0; end;
         p     = input('roll-rate [rad/s], default = 0: ');     if isempty(p)    , p     = 0; end;
         q     = input('pitch-rate [rad/s], default = 0: ');    if isempty(q)    , q     = 0; end;
         r     = input('yaw-rate [rad/s], default = 0: ');      if isempty(r)    , r     = 0; end;
         psi   = input('yaw-angle [rad], default = 0: ');       if isempty(psi)  , psi   = 0; end;
         theta = input('pitch-angle [rad], default = 0: ');     if isempty(theta), theta = 0; end;
         phi   = input('roll-angle [rad], default = 0: ');      if isempty(phi)  , phi   = 0; end;
         xe    = input('x-coordinate [m], default = 0: ');      if isempty(xe)   , xe    = 0; end;
         ye    = input('y-coordinate [m], default = 0: ');      if isempty(ye)   , ye    = 0; end;
         H     = input('altitude [m], default = 0: ');          if isempty(H)    , H     = 0; end;

         xinco  = [V alpha beta p q r psi theta phi xe ye H]';

         % Ask if state vector is correct. If not, re-enter, or go back
         % to main menu. If the length of the state vector is too short,
         % because the user has pressed <ENTER> without giving a number
         % for one or more of the states, the state-vector must be
         % re-entered by the user too.
         % ---------------------------------------------------------------
         disp(' ');

         if length(xinco) == 12    % xinco has right number of elements
            clc
            disp('Current definition of state vector:');
            xinco

            answ = input('Is this correct (y/[n])? ','s');
            if isempty(answ)
               answ = 'n';
            end
            if answ == 'y'
               ok1 = 1;            % leave loop for entering xinco
            else
               disp(' ');
               answ1 = input('Proceed with manual definition ([y]/n)? ','s');
               if isempty(answ1)
                  answ1 = 'y';
               end
               if answ1 == 'n'
                  proceed = 0;     % do not proceed with manual definition
               end
            end

         else                      % length of xinco is NOT right
            clc
            disp('State vector is too short! You probably have pressed');
            disp('<ENTER> without entering a value for one or more states.');
            disp(' ');

            answ1 = input('Proceed with manual definition ([y]/n)? ','s');
            if isempty(answ1)
               answ1 = 'y';
            end
            if answ1 == 'n'
               proceed = 0;     % do not proceed with manual definition
            else
               disp(' ');
               disp('Please re-enter state vector.');
            end

            disp(' ');
            disp('<<< Press a key >>>');
            pause
            clc
         end
      end

      % While-loop for correctly entering the aerodynamic inputs.
      % ---------------------------------------------------------
      ok1 = 0;        % If ok1 == 1, the while-loop for entering uaero0
                      % will be quitted; proceed with definition of uprop0,
                      % unless user has chosen to quit manual definition
                      % of user point (then proceed == 0).

      while ok1 ~= 1 & proceed == 1

         clc
         disp('Aerodynamic inputs: [deltae deltaa deltar deltaf]''');
         disp(' ');
         deltae = input('elevator angle [rad], default = 0: ');
         if isempty(deltae), deltae = 0; end;
         deltaa = input('ailerons deflection [rad], default = 0: ');
         if isempty(deltaa), deltaa = 0; end;
         deltar = input('rudder angle [rad], default = 0: ');
         if isempty(deltar), deltar = 0; end;
         deltaf = input('flap angle [rad], default = 0: ');
         if isempty(deltaf), deltaf = 0; end;

         uaero0 = [deltae deltaa deltar deltaf]';

         % Ask if aerodynamic inputvector is correct. If not, re-enter. If
         % the length of the aerodynamic inputvector is too short because
         % the user has pressed <ENTER> without giving a number for one or
         % more inputs, this inputvector must be re-entered by the user.
         % ---------------------------------------------------------------
         disp(' ');

         if length(uaero0) == 4  % uaero0 has right number of elements
            clc
            disp('Current definition of inputvector to aerodynamic model:');
            uaero0

            answ = input('Is this correct (y/[n])? ','s');
            if isempty(answ)
               answ = 'n';
            end
            if answ == 'y'
               ok1 = 1;           % leave while-loop to define uaero0
            else
               disp(' ');
               answ1 = input('Proceed with manual definition ([y]/n)? ','s');
               if isempty(answ1)
                  answ1 = 'y';
               end
               if answ1 == 'n'
                  proceed = 0;     % do not proceed with manual definition
               end
            end

         else                      % length of uaero0 is NOT right
            clc
            disp('Inputvector is too short! You probably have pressed');
            disp('<ENTER> without entering a value for one or more inputs.');
            disp(' ');

            answ1 = input('Proceed with manual definition ([y]/n)? ','s');
            if isempty(answ1)
               answ1 = 'y';
            end
            if answ1 == 'n'
               proceed = 0;     % do not proceed with manual definition
            else
               disp(' ');
               disp('Please re-enter state vector.');
            end

            disp(' ');
            disp('<<< Press a key >>>');
            pause
         end
      end

      % While-loop for correctly entering the engine inputvector.
      % ---------------------------------------------------------
      ok1 = 0;        % If ok1 == 1, the while-loop for entering uprop0
                      % will be quitted, and definitions of xinco, uaero0, and
                      % uprop0 entered so far will be used in linearization,
                      % unless the user already has specified that the
                      % manual operating point definition must be stopped,
                      % in which case proceed == 0.

      while ok1 ~= 1 & proceed == 1

         clc
         disp('Engine inputs: [n pz]''');
         disp(' ');
         n  = input('engine speed [RPM], default = 2000: ');    if isempty(n), n = 2000; end;
         pz = input('manifold pressure ["Hg], default = 20: '); if isempty(pz), pz = 20; end;

         uprop0 = [n pz]';

         % Ask if engine-inputvector is correct. If not, re-enter. If
         % the length of the engine-inputvector is too short because
         % the user has pressed <ENTER> without giving a number for one or
         % more inputs, this inputvector must be re-entered by the user.
         % ---------------------------------------------------------------
         disp(' ');

         if length(uprop0) == 2       % uprop0 has right length
            clc
            disp('Current definition of inputvector to engine model:');
            uprop0

            answ = input('Is this correct (y/[n])? ','s');
            if isempty(answ)
               answ = 'n';
            end
            if answ == 'y'
               ok1 = 1;
            else
               disp(' ');
               answ1 = input('Proceed with manual definition ([y]/n)? ','s');
               if isempty(answ1)
                  answ1 = 'y';
               end
               if answ1 == 'n'
                  proceed = 0;     % do not proceed with manual definition
               end

            end

         else                      % length of uprop0 is NOT right
            clc
            disp('Inputvector is too short! You probably have pressed');
            disp('<ENTER> without entering a value for one or more inputs.');
            disp(' ');

            answ1 = input('Proceed with manual definition ([y]/n)? ','s');
            if isempty(answ1)
               answ1 = 'y';
            end
            if answ1 == 'n'
               proceed = 0;     % do not proceed with manual definition
            else
               disp(' ');
               disp('Please re-enter state vector.');
            end

            disp(' ');
            disp('<<< Press a key >>>');
            pause
         end
      end

      % If user has not chosen to leave manual definition of operating point,
      % the variable proceed will still be equal to one. Then, it is right
      % to leave operating point definition, assuming that the current de-
      % finition is correct. Otherwise, go back to main menu (ok ~= 1!).
      % ---------------------------------------------------------------------
      if proceed == 1
         ok = 1;
      end

   elseif opt == 3                     % USE OPERATING POINT FROM WORKSPACE
                                       % ----------------------------------
      clc
      if exist('xinco') == 0 | exist('uaero0') == 0 | exist('uprop0') == 0
         % Currently, no operating point has been defined in the Matlab
         % workspace. Display error message and return to main menu.
         % ------------------------------------------------------------
         clc
         disp('ACLIN expects the following variables to be present in the');
         disp('Matlab workspace:');
         disp(' ');
         disp('   xinco = state vector in operating point');
         disp('   uaero0= vector with aerodynamic control inputs');
         disp('   uprop0= vector with engine control inputs');
         disp(' ');
         disp('At least one of these vectors is currently not present, so');
         disp('linearization cannot proceed!');
         disp(' ');
         disp('<<<Press a key to return to main menu>>>');
         pause
      else
         % Ask if current definition of the operating point is correct.
         % If not, program will return to main menu.
         % ------------------------------------------------------------
         clc
         disp('Current definition of operating point (xinco = states,');
         disp('uaero0 = aerodynamic inputs, uprop0 = engine inputs):');
         xinco
         uaero0
         uprop0
         answ = input('Is this correct (y/[n])? ','s');
         if isempty(answ) | answ ~= 'y'
            disp(' ');
            disp('<<<Press a key to return to main menu>>>');
            pause
         else
            ok = 1;  % If current definition is ok, proceed with
                     % linearization
         end
      end

   elseif opt == 4   % RUN ACTRIM TO DETERMINE STEADY-STATE TRIMMED-FLIGHT
                     % CONDITION. Type HELP ACTRIM for more details.
                     % ---------------------------------------------------

      % ACTRIM computes xinco, xdot0, uprop0, and uaero0. Before calling
      % ACTRIM, these variables are cleared to make sure that they will not
      % become mixed-up with the results of ACTRIM if they already exist.
      % To prevent other variables, used in ACLIN from intervening with
      % ACTRIM, all variables will be saved to a temporary file ACLIN.TMP.
      % The worspace will be cleared before starting ACTRIM. If ACTRIM has
      % finished, the old variables will be retrieved from ACLIN.TMP, and
      % the temporary file will be deleted.
      % ---------------------------------------------------------------------
      clear uaero0 uprop0 xinco xdot0 % delete operating point definition
                                      %            from workspace (if present)
                                      
      save aclin.tmp            % save remaining variables to temporary file
      clear                     % ... and clear workspace

      actrim                    % run ACTRIM

      load aclin.tmp -mat       % retrieve variables from temporary file
      delete('aclin.tmp');

      % Operating point definition is valid only if ACTRIM actually has
      % computed something. If option 'Quit' was selected in ACTRIM, the
      % workspace does not contain a valid definition of the operating
      % point; ok = 1 (quit operating point definition loop) is therefore
      % defined only if xinco, uprop0, and uaero0 exist.
      % -----------------------------------------------------------------
      if exist('xinco') == 0 | exist('uaero0') == 0 | exist('uprop0') == 0
         disp(' ');
         disp('No operating point defined!');
         disp(' ');
         disp(' ');
         disp('<<< Press a key to return to main menu >>>');
         pause
      else
         ok = 1;
      end
   else                                                             % QUIT
                                                                    % ----
      ok   = 1;
      skip = 1;  % Return to end of program rightaway when quitting.

   end

end % of operating-point definition.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% if skip ~= 1   % If not quitting from ACLIN, proceed with program.
               % Else, go to end of program rightaway!

   disp(' ');
   disp('<<< Press a key to proceed with model definition >>>');
   pause

   % Define parameters for aircraft model:
   % =====================================
   %
   % AM, EM, GM1, and GM2: matrices, containing parameters for the aircraft
   %                       model. See also the help text and source code for
   %                       MODBUILD.M (contained in the directory AIRCRAFT).
   %
   % The other variables will either be explained when they are used, or are
   % self-explaining (at least if you know something about aircraft stabi-
   % lity and control).
   %
   % Check if these variables already have been defined in the Matlab work-
   % space (by ACTRIM). If not, run DATLOAD to retrieve them from file.
   % -----------------------------------------------------------------------
   if exist('AM')==0 | exist('EM')==0 | exist('GM1')==0 | exist('GM2')==0
       h=newMsgBox(['First, the model parameters need to be retrieved from file ', ...
                    '(e.g. AIRCRAFT.DAT). Click ''OK'' or press Enter to continue.']);
       uiwait(h);
       datload('aircraft');
   end

   % If model parameters are still not present in the workspace,
   % e.g. due to an incorrect datafile, force an abort of ACLIN.
   % ------------------------------------------------------------
   if exist('AM')==0 | exist('EM')==0 | exist('GM1')==0 | exist('GM2')==0
       error(['ERROR: the model parameters are still not present in the workspace! ', ...
                    'Aborting ACLIN.']);
   end
xfix = 1;


   xg_vett =1.0306 %[0.9492 0.9805 1.0055 1.0306 1.0556 1.0806 1.1057 1.1307 1.1808 1.2434];
   CM0_vett =-0.09731 %-[0.108 0.102 0.09969 0.09731 0.09496 0.09262 0.09024 0.08789 0.08316 0.07728];
   CMalpha_vett = - 1.228%-[1.589 1.450 1.339 1.228 1.117 1.007 0.8957 0.7848 0.5631 0.2861];
   CMalpha2_vett = 0.2354 %[0.2401 0.2386 0.2367 0.2354 0.2335 0.234 0.2313 0.2299 0.227 0.2238];
   CMq_vett = -15.76 %-[16.57 16.25 16.00 15.76 15.52 15.30 15.08 14.86 14.45 13.98];
   CMdeltae_vett =- 0.7522%-[0.7678 0.7618 0.757 0.7522 0.7473 0.7424 0.7377 0.7328 0.7232 0.7111];
   CLq_vett = 7.77%[8.492 8.214 7.993 7.77 7.548 7.326 7.104 6.882 6.438 5.884];
   CYr_vett =0.2340 %[0.2393 0.2372 0.2356 0.2340 0.2323 0.2307 0.2290 0.2274];
   CYp_vett = -0.0839;
   Clb_vett = -0.0424;
   Clp_vett = -0.447;
   Clr_vett = 0.0759;
   Cnb_vett = 0.06935%[0.07426 0.07237 0.07087 0.06935 0.06785 0.06634 0.06433 0.06333];
   Cnp_vett = -0.02968 %-[0.03027 0.03006 0.02987 0.02968 0.02951 0.02933 0.02914 0.02897];
   Cnr_vett = -0.21643 %-[0.22315 0.22055 0.2185 0.21643 0.2139 0.2111 0.2082 0.2054];
   Cndr_vett = -0.0273 %-[0.0278 0.0276 0.0274 0.0273 0.0271 0.0269 0.0268 0.0266];
   autovalRe = eye(0);
   autovalIm = eye(0);
   dampingCP = [];
   dampingPH = [];
   pulseCP = [];
   pulsePH = [];
   TCP = [];
   TPH = [];
   tmezziCP = [];
   tmezziPH = [];
  % xdef = [1 2 5 8];
   xdef = [3 4 6 9];
   

   for vett=1:1:(length(CM0_vett))%-2)
       AM(5,1) = CM0_vett(vett);
       AM(5,2) = CMalpha_vett(vett);
       AM(5,3) = CMalpha2_vett(vett);
       AM(5,9) = CMq_vett(vett);
       AM(5,11) = CMdeltae_vett(vett);
       AM(3,9) = CLq_vett(vett);
       AM(2,10) = CYr_vett(vett);
       AM(2,8) = CYp_vett;
       AM(4,5) = Clb_vett;
       AM(4,8) = Clp_vett;
       AM(4,10) = Clr_vett;
       AM(6,5) = Cnb_vett(vett);
       AM(6,8) = Cnp_vett(vett);
       AM(6,10) = Cnr_vett(vett);
       AM(6,14) = Cndr_vett(vett);
       
      

   % Proceed with linearization.
   % ===========================

   % Build initial inputvector to the system (contains both aerodynamic
   % and engine inputs). Note: wind velocities and accelerations (inclu-
   % ding atmospheric turbulence) have been set to zero!!!
   % -------------------------------------------------------------------
   uinco = [uaero0; uprop0; 0; 0; 0; 0; 0; 0];



   % Apply Simulink routine LINMOD to nonlinear aircraft model, contained
   % in the Simulink model 'sysname.mdl'. Note: contrary to ACTRIM, the
   % linearization tool LINMOD is a standard Simulink function. Hence, it
   % is not necessary to initialize the system at this place by calling
   % it once with [sys,x0] = sysname([],[],[],0)!
   % --------------------------------------------------------------------
   [Aac, Bac, Cac, Dac] = linmod(sysname,xinco,uinco);



   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


   % Select a subset of the states and inputs. Note: the remaining states
   % and inputs are omitted from the state equation by simply eliminating
   % the corresponding elements from the state matrices. The more elements
   % you neglect, the larger the resulting error will be.
   %
   % ACLIN doesn't provide the option to choose a subset of the output
   % equations, because the exact definition of the outputvector from the
   % Simulink system which contains the nonlinear aircraft model is not
   % known (in general). The full twelfth-order linear aircraft model
   % does include the output equations y = Cac*x + Dac*u, but the exact
   % definition of the matrices Cac and Dac, and the outputvector y can be
   % determined only by checking the corresponding nonlinear Simulink
   % model!
   % =====================================================================

   % While-loop in which the user must specify the element numbers of the
   % STATES that should be used for the linear model. These element num-
   % bers are stored in the vector xdef (state definition).
   %
   % The flag 'allstates' is set to one if user wants to maintain defini-
   % tion of state vector. If not, allstates = 0.
   % --------------------------------------------------------------------
   allstates = 0;
   ok = 0;

         xdef = xdef;
         allstates = 1;                   % Set all states flag.


   % While-loop in which the user must specify the element numbers of the
   % INPUTS that should be used for the linear model. These element num-
   % bers are stored in the vector udef (input definition). Note: wind
   % and turbulence inputs (disturbances) are selected separately.
   %
   % The flag 'allinputs' is set to one if the user wants to maintain
   % the full inputvector, INCLUDING atmospheric disturbances. If not,
   % allinputs = 0.
   % --------------------------------------------------------------------


         udef = [1 2 3 4 5 6];



   
        allinputs = 0;  

   % If user has selected all states AND all outputs, including the
   % atmospheric disturbances (wind & turbulence), there is no need
   % to determine new, simplified matrices for the linearized model,
   % because Aac_s == Aac and Bac_s == Bac in that case. If the user
   % has selected a subset of x and u, or if he has defined the vectors
   % xdef and udef such that the order of the states and/or inputs is
   % shuffled, simplified versions of Aac and Bac will be created.
   %
   % First check if user has specified full state and inputvector
   % (unshuffled) for linearized model. If not, define Aac_s and Bac_s.
   % ------------------------------------------------------------------


      % Determine new state matrix Aac_s, depending upon element numbers
      % selected above.
      % ----------------------------------------------------------------
      for ii = 1:length(xdef)             % ii = row number
         for jj = 1:length(xdef)          % jj = column number
            Aac_s(ii,jj) = Aac(xdef(ii),xdef(jj));
         end
      end

      % Determine new state matrix Bac_s, depending upon element numbers
      % selected above.
      % ----------------------------------------------------------------
      for ii = 1:length(xdef)             % ii = row number
         for jj = 1:length(udef)          % jj = column number
            Bac_s(ii,jj) = Bac(xdef(ii),udef(jj));
         end
      end


lambda = eig(Aac_s);
autovalRe = [autovalRe real(lambda)];
autovalIm = [autovalIm imag(lambda)];
pulseCP(vett) = sqrt((imag(lambda(1)))^2 + (real(lambda(1)))^2);
pulsePH(vett) = sqrt((imag(lambda(3)))^2 + (real(lambda(3)))^2);
dampingCP(vett) = -real(lambda(1))/pulseCP(vett);
dampingPH(vett) = -real(lambda(3))/pulsePH(vett);
   
   TCP(vett) = 2*pi/imag(lambda(1));
   TPH(vett) = 2*pi/imag(lambda(3));
   tmezziCP(vett) = log(2)/real(lambda(1));
   tmezziPH(vett) =log(2)/real(lambda(3)) ;
   
   
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   % Create a text matrix with information about the linearized model.
   % Note: the function NUM2STR2 is just a slightly changed version
   % of NUM2STR. Type HELP NUM2STR2 at the Matlab command line  for
   % more info.
   % =================================================================



   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


   % Results can now be saved to a file, which will be filled with the
   % state space matrices Aac, Bac, Cac, Dac of the linearized aircraft
   % model, and the information matrix lindef. Moreover, the user will
   % be asked if the operating point, defined by xinco, uaero0, and uprop0
   % must be saved to the same file too. In that case, the information
   % matrix trimdef will be saved too if it exists (see ACTRIM for more
   % info).
   %
   % Note: files with linearized aircraft model have extension .LIN!
   % ==================================================================


      % Destination directory for storing the results is the default FDC 
      % data-directory, which is obtained from the function DATADIR.M. 
      % If that directory does not exist, the current directory will be 
      % used instead.
      % ----------------------------------------------------------------


      % Go to default directory if that directory exists (if not, start
      % save-routine from the current directory).
      % ---------------------------------------------------------------

      % Obtain filename and path
      % ------------------------ 


      % Ask if operating point needs to be included to the .LIN file. 
      % If it does, append xinco, uaero0, uprop0 and (if present) 
      % trimdef to the saved file.
      % -------------------------------------------------------------

         
         % If operating point has been loaded from .TRI file or computed by
         % directly calling ACTRIM, the definition of the trimmed flight-
         % condition is stored in the text-matrix trimdef. If the operating
         % point is defined in another way, this matrix does not exist. If
         % trimdef exists, it will be saved along with the trimmed flight-
         % condition itself.
         % -----------------------------------------------------------------


      % Display file location
      % ---------------------


   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   % Display results for user.
   % =========================


   end
autovalRe
disp(' ');
autovalIm
disp('Ready.');
disp(' ');

% Clear variables which are not needed anymore
% --------------------------------------------
% clear V alpha beta p q r psi theta phi xe ye H deltae deltaa deltar
% clear deltaf n pz ok ok1 ok2 proceed answ answ1
% clear opt skip sysname uinco udef xdef allstates allinputs ii jj h
% clear t t1 t2 t3 dirname filename savecmmnd defdir currentdir


%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: December 31, 2004.
%-----------------------------------------------------------------------